/*
   Scott's AKE Client/Server testbed

   See http://eprint.iacr.org/2002/164

   Compile as 
   cl /O2 /GX /DZZNS=5 ake6.cpp zzn6.cpp ecn3.cpp zzn3.cpp big.cpp zzn.cpp ecn.cpp miracl.lib
   using COMBA build

   The required file mnt.ecs is created from a curve generated by the mnt 
   utility, and created by the cm utility. For convenience the value of 
   (p^2-p+1)/q has been manually appended to this file.

   NOTE: Irreducible polynomial MUST be of the form x^6+n
   Use the irred utility

   Modified to prevent sub-group confinement attack

   NOTE: Key exchange bandwidth could be reduce by two-thirds using ideas from
         "Doing more with Fewer Bits", Brouwer, Pellikaan & Verheul, Asiacrypt 
         '99

*/

#include <iostream>
#include <fstream>
#include <string.h>
#include "ecn.h"
#include <ctime>
#include "ecn3.h"
#include "zzn6.h"

// cofactor - number of points on curve=CF.q

#define CF 2  

using namespace std;

Miracl precision(5,0); 

// Using SHA-1 as basic hash algorithm

#define HASH_LEN 20

//
// Define one or the other of these
//
// Which is faster depends on the I/M ratio - See imratio.c
// Roughly if I/M ratio > 16 use PROJECTIVE, otherwise use AFFINE
//

//#define AFFINE
#define PROJECTIVE

//
// Tate Pairing Code
//
// Extract ECn point in internal ZZn format
//

void extract(ECn& A,ZZn& x,ZZn& y)
{ 
    x=(A.get_point())->X;
    y=(A.get_point())->Y;
}

void extract(ECn& A,ZZn& x,ZZn& y,ZZn& z)
{ 
    big t;
    x=(A.get_point())->X;
    y=(A.get_point())->Y;
    t=(A.get_point())->Z;
    if (A.get_status()!=MR_EPOINT_GENERAL) z=1;
    else                                   z=t;
}

//
// Line from A to destination C. Let A=(x,y)
// Line Y-slope.X-c=0, through A, so intercept c=y-slope.x
// Line Y-slope.X-y+slope.x = (Y-y)-slope.(X-x) = 0
// Now evaluate at Q -> return (Qy-y)-slope.(Qx-x)
//

ZZn6 line(ECn& A,ECn& C,ZZn& slope,ZZn3& Qx,ZZn3& Qy)
{
     ZZn6 w;
     ZZn3 nn=Qx;
     ZZn x,y,z,t;

#ifdef AFFINE
     extract(A,x,y);
     nn-=x; nn*=slope; nn+=y;
     w.set(-nn,Qy);
#endif
#ifdef PROJECTIVE
     extract(A,x,y,z);
     x*=z; t=z; z*=z; z*=t;
     x*=slope; t=slope*z;
     nn*=t; nn-=x; t=z;
     extract(C,x,x,z);
     nn+=(z*y); t*=z;
     w.set(nn,-Qy*t);

#endif
     return w;
}


//
// Add A=A+B  (or A=A+A) 
// Return line function value
//

ZZn6 g(ECn& A,ECn& B,ZZn3& Qx,ZZn3& Qy)
{
    ZZn  lam;
    ZZn6 u;
    big ptr;
    ECn P=A;

// Evaluate line from A
    ptr=A.add(B);
    if (ptr==NULL)    return (ZZn6)1;
    else lam=ptr;

    if (A.iszero())   return (ZZn6)1; 

    return line(P,A,lam,Qx,Qy);
  
}

//
// Tate Pairing - note denominator elimination has been applied
//
// P is a point of order q. Q(x,y) is a point of order m.q. 
// Note that P is a point on the curve over Fp, Q(x,y) a point on the 
// twisted curve over the extension field Fp^3
//

BOOL fast_tate_pairing(ECn& P,ZZn3& Qx,ZZn3& Qy,Big& q,Big &cf,ZZn3& r)
{ 
    int i,j,n,nb,nbw,nzs;
    ECn A,P2,t[16];
    ZZn6 w,hc,z2n,zn[16],b[2],res;
    Big a[2];

    res=zn[0]=1;  

    t[0]=P2=A=P;
    z2n=g(P2,P2,Qx,Qy);     // P2=P+P
//
// Build windowing table
//
    for (i=1;i<16;i++)
    {           
        hc=g(A,P2,Qx,Qy); 
        t[i]=A;         
        zn[i]=z2n*zn[i-1]*hc;
    }
    A=P;    // reset A

/* Left to right method  */
    nb=bits(q);
    for (i=nb-2;i>=0;i-=(nbw+nzs))
    {
        n=window(q,i,&nbw,&nzs,5);  // standard MIRACL windowing
        for (j=0;j<nbw;j++)
        {
            res*=res;           
            res*=g(A,A,Qx,Qy); 
        }
        if (n>0)
        {
            res*=zn[n/2];     
            res*=g(A,t[n/2],Qx,Qy);
        }  
        for (j=0;j<nzs;j++) 
        {
            res*=res; 
            res*=g(A,A,Qx,Qy); 
        }  
        if (res.iszero()) return FALSE;
    }
  
    if (!A.iszero() || res.iszero()) return FALSE;

    w=res;                          
    w.powq();
    res*=w;                        // ^(p+1)

    w=res;
    w.powq(); w.powq(); w.powq();
    res=w/res;                     // ^(p^3-1)

// res=pow(res,cf);               // ^(p*p-p+1)/q

// exploit the clever "trick" for a half-length exponentiation!

    w=res.powq();
    res.powq(); res*=res; // res*=res;  // res=pow(res,CF);
    
    if (cf<0) res/=powu(w,-cf);
    else res*=powu(w,cf);

    r=real(res);
    if (r==(ZZn3)1) return FALSE;
    return TRUE;            
}

//
// ecap(.) function
//

BOOL ecap(ECn& P,ECn3& Q,Big& order,Big& cf,ZZn3& r)
{
    BOOL Ok;
    ECn PP=P;
    ZZn3 Qx,Qy;
    int qnr=-get_mip()->cnr;

    normalise(PP);
    Q.get(Qx,Qy);

// untwist    
    Qx=Qx/qnr;
    Qy=tx(Qy);
    Qy=Qy/(qnr*qnr);
 
    Ok=fast_tate_pairing(PP,Qx,Qy,order,cf,r);

    if (Ok) return TRUE;
    return FALSE;
}

//
// Hash functions
// 

Big H1(char *string)
{ // Hash a zero-terminated string to a number < modulus
    Big h,p;
    char s[HASH_LEN];
    int i,j; 
    sha sh;

    shs_init(&sh);

    for (i=0;;i++)
    {
        if (string[i]==0) break;
        shs_process(&sh,string[i]);
    }
    shs_hash(&sh,s);
    p=get_modulus();
    h=1; j=0; i=1;
    forever
    {
        h*=256; 
        if (j==HASH_LEN)  {h+=i++; j=0;}
        else         h+=s[j++];
        if (h>=p) break;
    }
    h%=p;
    return h;
}

Big H2(ZZn3 x)
{ // Hash an Fp3 to a big number
    sha sh;
    ZZn u,v,w;
    Big a,h,p,xx[3];
    char s[HASH_LEN];
    int i,j,m;

    shs_init(&sh);
    x.get(u,v,w);
    xx[0]=u; xx[1]=v; xx[2]=w;
    for (i=0;i<3;i++)
    {
        a=xx[i];
        while (a>0)
        {
            m=a%256;
            shs_process(&sh,m);
            a/=256;
        }
    }
    shs_hash(&sh,s);
    h=from_binary(HASH_LEN,s);
    return h;
}

// Hash and map a Server Identity to a curve point E_(Fp3)

ECn3 hash_and_map3(char *ID)
{
    int i;
    ECn3 S;
    ZZn3 X;
 
    Big x0=H1(ID);
    forever
    {
        x0+=1;
        X.set2((ZZn)x0);
        if (!S.set(X)) continue;

        break;
    }

//    cout << "S= " << S << endl;
    return S;
}     

// Hash and map a Client Identity to a curve point E_(Fp) of order q

ECn hash_and_map(char *ID)
{
    ECn Q;
    Big x0=H1(ID);

    while (!Q.set(x0,x0)) x0+=1;
    Q*=CF;
    return Q;
}

int main()
{
    ifstream common("mnt.ecs");      // MNT elliptic curve parameters
    miracl* mip=&precision;
    ECn Alice,Bob,sA,sB;
    ECn3 B6,Server,sS;
    ZZn3 sp,ap,bp,res;
    Big a,b,s,ss,p,q,x,y,B,cf,cfp;
    int i,bits,A;
    time_t seed;

    common >> bits;
    mip->IOBASE=16;
    common >> p;
    common >> A;
    common >> B >> q >> cf;
   
    init_zzn6(p);

    cout << "Initialised... " << endl;
    cout << "p%24= " << p%24 << endl;
    cout << "cnr= " << mip->cnr << endl;
//cout << (Big)(mip->sru) << endl;
    time(&seed);
    irand((long)seed);

#ifdef AFFINE
    ecurve(A,B,p,MR_AFFINE);
#endif
#ifdef PROJECTIVE
    ecurve(A,B,p,MR_PROJECTIVE);
#endif

    cfp=cf-CF*p;  // ~ (t-1)

    mip->IOBASE=16;
    mip->TWIST=TRUE;   // map Server to point on twisted curve E(Fp3)

    ss=rand(q);    // TA's super-secret 

    cout << "Mapping Server ID to point" << endl;
    Server=hash_and_map3("Server");


    cout << "Mapping Alice & Bob ID's to points" << endl;
    Alice=hash_and_map("Alice");
    Bob=  hash_and_map("Robert");

    cout << "Alice, Bob and the Server visit Trusted Authority" << endl; 

    sS=ss*Server; 
    sA=ss*Alice; 
    sB=ss*Bob; 

    cout << "Alice and Server Key Exchange" << endl;

    a=rand(q);   // Alice's random number
    s=rand(q);   // Server's random number

//    for (i=0;i<10000;i++)
    if (!ecap(sA,Server,q,cfp,res)) cout << "Trouble" << endl;
    if (powl(res,q)!=(ZZn3)1)
    {
        cout << "Wrong group order - aborting" << endl;
        exit(0);
    }
    ap=powl(res,a);

    if (!ecap(Alice,sS,q,cfp,res)) cout << "Trouble" << endl;
    if (powl(res,q)!=(ZZn3)1)
    {
        cout << "Wrong group order - aborting" << endl;
        exit(0);
    }
    sp=powl(res,s);

    cout << "Alice  Key= " << H2(powl(sp,a)) << endl;
    cout << "Server Key= " << H2(powl(ap,s)) << endl;

    cout << "Bob and Server Key Exchange" << endl;

    b=rand(q);   // Bob's random number
    s=rand(q);   // Server's random number

    if (!ecap(sB,Server,q,cfp,res)) cout << "Trouble" << endl;
    if (powl(res,q)!=(ZZn3)1)
    {
        cout << "Wrong group order - aborting" << endl;
        exit(0);
    }
    bp=powl(res,b);

    if (!ecap(Bob,sS,q,cfp,res)) cout << "Trouble" << endl;
    if (powl(res,q)!=(ZZn3)1)
    {
        cout << "Wrong group order - aborting" << endl;
        exit(0);
    }
    sp=powl(res,s);

    cout << "Bob's  Key= " << H2(powl(sp,b)) << endl;
    cout << "Server Key= " << H2(powl(bp,s)) << endl;

    return 0;
}

