/*

Example Elliptic Curve Diffie-Hellman program for 8-bit constrained environments. Uses point compression.
Stack-only memory allocation

Use with this mirdef.h header (for a PC using MS C)

#define MR_LITTLE_ENDIAN
#define MIRACL 8
#define mr_utype char
#define MR_IBITS 32
#define MR_LBITS 32
#define mr_unsign32 unsigned int
#define mr_dltype short
#define MR_STATIC 20
#define MR_ALWAYS_BINARY
#define MR_NOASM
#define MR_STRIPPED_DOWN
#define MR_GENERIC_MT
#define MAXBASE ((mr_small)1<<(MIRACL-1))
#define MR_COMBA 20
#define MR_SPECIAL
#define MR_BITSINCHAR 8
#define MR_SHORT_OF_MEMORY
#define MR_SMALL_EWINDOW
#define MR_NOKOBLITZ

Build the library from these modules (Example using MS C compiler)

mex 20 c mrcomba
rem In a real 8-bit setting, a suitable assembly language .mcs file would be used. Here we use C.
cl /c /O2 /W3 mrcore.c
cl /c /O2 /W3 mrarth0.c
cl /c /O2 /W3 mrarth1.c
cl /c /O2 /W3 mrio1.c
cl /c /O2 /W3 mrbits.c
cl /c /O2 /W3 mrcurve.c
cl /c /O2 /W3 mrsroot.c
cl /c /O2 /W3 mrjack.c
cl /c /O2 /W3 mrlucas.c
cl /c /O2 /W3 mrsmall.c
cl /c /O2 /W3 mrarth2.c
cl /c /O2 /W3 mrmonty.c
cl /c /O2 /W3 mrcomba.c
cl /c /O2 /W3 mrxgcd.c
cl /c /O2 /W3 mrebrick.c

rem
rem Create library 'miracl.lib'
del miracl.lib

lib /OUT:miracl.lib mrio1.obj mrmonty.obj mrcomba.obj mrxgcd.obj
lib /OUT:miracl.lib miracl.lib mrbits.obj mrarth2.obj mrlucas.obj mrjack.obj
lib /OUT:miracl.lib miracl.lib mrarth0.obj mrarth1.obj mrcore.obj mrebrick.obj
lib /OUT:miracl.lib miracl.lib mrcurve.obj mrsroot.obj mrsmall.obj
del mr*.obj

rem Create the program

cl /O2 ecdhp8.c miracl.lib

For Atmel AVR (atmega128) use

#define MR_LITTLE_ENDIAN
#define MIRACL 8
#define mr_utype char
#define MR_IBITS 16  
#define MR_LBITS 32
#define mr_unsign32 unsigned long
#define mr_dltype int 
#define mr_qltype long
#define MR_STATIC 20
#define MR_ALWAYS_BINARY
#define MR_STRIPPED_DOWN
#define MR_GENERIC_MT
#define MAXBASE ((mr_small)1<<(MIRACL-1))
#define MR_COMBA 20
#define MR_SPECIAL
#define MR_NOASM
#define MR_BITSINCHAR 8
#define MR_NO_STANDARD_IO
#define MR_NO_FILE_IO
#define MR_SIMPLE_IO
#define MR_SMALL_EWINDOW
#define MR_AVR

Note this last line is added manually - it will not be added by config

and execute

mex 20 avr mrcomba

Note that reading data from program memory (NVM) is a little complex on the AVR!

*/

#include <stdio.h>
#include "miracl.h"

#define CURVE_BITS 160

/* SCOTT p160 bit elliptic curve Y^2=X^3-3X+157 modulo 2^160-2^112+2^64+1 
   Here is stored P, B, the group order q, and the generator G(x,y)       */

#ifdef MR_AVR
__attribute__((__progmem__))
#endif 
static const mr_small rom[]=
{
0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
0x9D,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0xC3,0x35,0x39,0xC7,0x6F,0xA3,0xC9,0x36,0x3A,0x59,0xFE,0xFF,0xFF,0xFF,0xFE,0xFF,0xFF,0xFF,0xFF,0xFF,    
0x1E,0xA1,0x35,0x4E,0xAD,0x6A,0x3E,0x37,0xAD,0x87,0x67,0xAC,0x4C,0xE1,0x68,0xEF,0x83,0x05,0x16,0xE1,
0x60,0x84,0x9E,0xF2,0x44,0xCE,0xF0,0x63,0x8B,0x5E,0x1E,0x76,0x8A,0xE4,0x02,0x6D,0x74,0x75,0x07,0x57};

#define WINDOW 4

/* 32 precomputed points based on fixed generator G(x,y)        */
/* (created using ebrick.c program with window size of 4)       */
/* NOTE: If MR_SPECIAL is defined in mirdef.h for this program, */
/* it MUST be defined for the build of ebrick.c as well         */

#ifdef MR_AVR
__attribute__((__progmem__))
#endif 
static const mr_small prom[]=
{
0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,
0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,
0x1e,0xa1,0x35,0x4e,0xad,0x6a,0x3e,0x37,0xad,0x87,0x67,0xac,0x4c,0xe1,0x68,0xef,0x83,0x5,0x16,0xe1,
0x60,0x84,0x9e,0xf2,0x44,0xce,0xf0,0x63,0x8b,0x5e,0x1e,0x76,0x8a,0xe4,0x2,0x6d,0x74,0x75,0x7,0x57,
0x72,0x67,0x7f,0x39,0x67,0x97,0xed,0xf0,0xea,0xf5,0x54,0xf9,0xf4,0x38,0x28,0x29,0x45,0x7b,0xa,0x2f,
0x48,0x64,0xb8,0xae,0xd7,0x22,0x28,0x7e,0x6a,0x1d,0x9,0x35,0xa7,0x7e,0xcd,0xa3,0x3a,0x9c,0x9b,0xf9,
0x44,0x5,0x89,0xd3,0xa6,0x55,0xd5,0xa7,0x55,0xb7,0x32,0xe5,0xa2,0x77,0x83,0x6c,0xb6,0xc3,0x4,0xae,
0xca,0x5e,0xa9,0xf,0x52,0x88,0x4f,0xf6,0xd6,0x40,0x2,0x5b,0x60,0x89,0x5c,0xd0,0x1f,0xa3,0x75,0x16,
0x73,0x98,0xb3,0x5a,0xa1,0xcf,0x39,0xcd,0x9a,0xf1,0x4c,0x5a,0x4,0xea,0xb5,0x23,0x79,0xc5,0x53,0xcb,
0xf1,0x8b,0x76,0x3d,0xc8,0x85,0x3d,0x12,0x5b,0x44,0x30,0x0,0xef,0x2c,0x40,0xb2,0xc9,0x69,0x67,0x43,
0xee,0x43,0x7f,0xa5,0x4,0xb7,0x9e,0xdf,0x10,0x64,0x9c,0x35,0x20,0x8e,0xf5,0xf1,0x11,0x17,0xd4,0x52,
0xf1,0x5c,0x81,0x34,0xea,0xa1,0x2c,0x43,0x16,0xd5,0x73,0xb0,0xa8,0x9b,0xa3,0xc4,0xc7,0x9c,0x19,0x6c,
0x1b,0x60,0xf7,0xb7,0x94,0x9a,0x11,0x80,0xd,0x5,0xb5,0xe1,0xdb,0x34,0xa5,0x98,0xcb,0xb4,0x54,0xf3,
0xf3,0xd8,0x8,0x8b,0x76,0x11,0x70,0xa2,0xa6,0xed,0x69,0x6d,0x5,0xd3,0x9a,0x6c,0xa,0xab,0x1d,0x56,
0x7a,0xa5,0xad,0xe3,0x98,0x6d,0xa2,0x45,0x2,0xfa,0x30,0x99,0x4a,0x74,0xfb,0x7,0x3b,0x26,0xa9,0xec,
0xc,0x23,0xf9,0x39,0x8a,0xac,0x3e,0x11,0xd1,0x2f,0xcb,0x17,0xbf,0x18,0x10,0xed,0x25,0xd2,0x56,0xda,
0xd3,0x7a,0x66,0x6d,0xed,0x9f,0xcf,0x13,0x86,0xed,0x88,0xe2,0xeb,0x55,0xa1,0x61,0x2d,0xc9,0x67,0xa0,
0xf8,0x8e,0x8d,0x6b,0x7a,0x6,0x91,0xa0,0x31,0xef,0x8a,0x94,0x3e,0x55,0x80,0x72,0x32,0xe6,0xa8,0x45,
0xc8,0x4f,0xc2,0x62,0x2f,0xd8,0xb2,0x36,0x43,0x51,0x6d,0x83,0x73,0xb0,0xfc,0xc6,0x0,0xf9,0x5,0x55,
0xf8,0xef,0x35,0x28,0xbb,0xf,0x23,0x95,0xf5,0x8,0x1c,0xfe,0xe,0xb1,0xe5,0x94,0xea,0x35,0x5b,0xe7,
0xc7,0x43,0xfd,0x1d,0xc8,0xe2,0xd1,0x5e,0xd1,0x2c,0x7e,0xcb,0x10,0x2,0xea,0x32,0xd7,0x16,0x4b,0x76,
0x27,0xaf,0x21,0x17,0xac,0x37,0xf4,0xe1,0x94,0xab,0xf5,0xb2,0xdc,0xb1,0x94,0x7a,0x11,0x6,0x2e,0x4,
0x81,0xc7,0xc5,0x54,0x84,0x57,0x20,0xa0,0x23,0xd7,0x6d,0xdf,0xb9,0x76,0xc9,0xfb,0x4e,0xb4,0xfb,0x9f,
0xd4,0x9c,0xea,0x1b,0x8b,0x81,0x45,0x70,0x2d,0xad,0xaa,0xbe,0x5b,0x50,0x78,0xc7,0x61,0x53,0xc2,0x15,
0x66,0x6b,0xa5,0x81,0x67,0xdf,0xfe,0xad,0xfa,0xe2,0xc5,0xbd,0x77,0xef,0xeb,0x5e,0xa5,0x60,0x37,0x90,
0xcc,0xa1,0x97,0x9,0xdf,0xb8,0x9,0xfc,0xe0,0x49,0x8a,0x91,0x93,0x29,0x18,0x4f,0x76,0xf8,0x80,0x96,
0x4b,0x69,0x7d,0x3,0xce,0xbb,0x9f,0x3d,0x94,0x54,0x16,0x9b,0xfb,0x25,0xb,0x76,0xc3,0xc6,0x41,0x9b,
0xda,0x1b,0xa5,0x2f,0xce,0x45,0xa9,0x3d,0x10,0x48,0xb8,0x23,0x22,0xa7,0xb9,0xce,0x45,0xc0,0x66,0x33,
0x5e,0xe6,0xba,0x62,0x5a,0x67,0xf5,0xfd,0x14,0xc8,0xfd,0x23,0xc9,0x33,0x6,0x4c,0x87,0xe7,0x6d,0xa0,
0x58,0x73,0x32,0x61,0x31,0xf7,0x84,0xf8,0xd3,0xe4,0x3b,0x9f,0xcc,0x9f,0x8c,0x45,0x96,0xfb,0xc5,0x58,
0xb1,0x6f,0x7f,0xe,0x40,0x2,0xdb,0x8a,0x95,0x90,0xdc,0xe4,0x62,0xe6,0xb8,0x54,0x16,0x17,0xe4,0x8a,
0x38,0x9c,0x61,0x8a,0xad,0xfc,0xef,0x78,0x29,0x20,0xc,0x3c,0x18,0xed,0xaf,0x19,0xfb,0xd4,0xf5,0xfb};

#define WORDS 20  /* Number of words per big variable 20*8 = 160 */

/* Note that in a real application a source of real random numbers would be required, to
   replace those generated by MIRACL's internal pseudo-random generator "bigbits"  
   Alternatively from a truly random and unguessable seed, use MIRACL's strong random 
   number generator */

/* Elliptic Curve Diffie-Hellman, using point compression to minimize bandwidth, 
   and precomputation to speed up off-line calculation */


int main()
{
    int ia,ib,promptr;
    epoint *PA,*PB;
    big A,B,p,a,b,pa,pb,key,x,y;
    ebrick binst;
    miracl instance;      /* create miracl workspace on the stack */

/* Specify base 16 here so that HEX can be read in directly without a base-change */

    miracl *mip=mirsys(&instance,WORDS*2,16); /* size of bigs is fixed */
    char mem_big[MR_BIG_RESERVE(10)];         /* we need 10 bigs... */
    char mem_ecp[MR_ECP_RESERVE(2)];          /* ..and two elliptic curve points */
 	memset(mem_big, 0, MR_BIG_RESERVE(10));   /* clear the memory */
	memset(mem_ecp, 0, MR_ECP_RESERVE(2));

    A=mirvar_mem(mip, mem_big, 0);       /* Initialise big numbers */
    B=mirvar_mem(mip, mem_big, 1);
    pa=mirvar_mem(mip, mem_big, 2);
    pb=mirvar_mem(mip, mem_big, 3);
    key=mirvar_mem(mip, mem_big, 4);
    x=mirvar_mem(mip, mem_big, 5);
    y=mirvar_mem(mip, mem_big, 6);
    a=mirvar_mem(mip, mem_big, 7);
    b=mirvar_mem(mip, mem_big, 8);
    p=mirvar_mem(mip, mem_big, 9);

    PA=epoint_init_mem(mip, mem_ecp, 0); /* initialise Elliptic Curve points */
    PB=epoint_init_mem(mip, mem_ecp, 1);

    irand(mip, 3L);                      /* change parameter for different random numbers */

    promptr=0;
    init_big_from_rom(p,WORDS,rom,100,&promptr);  /* Read in prime modulus p from ROM   */
    init_big_from_rom(B,WORDS,rom,100,&promptr);  /* Read in curve parameter B from ROM */
                                                 /* don't need q or G(x,y) (we have precomputed table from it) */

    convert(mip,-3,A);                           /* set A=-3 */

/* Create precomputation instance from precomputed table in ROM */

    ebrick_init(&binst,prom,A,B,p,WINDOW,CURVE_BITS);

/* offline calculations */

    bigbits(mip,CURVE_BITS,a);  /* A's random number */
    ia=mul_brick(mip,&binst,a,pa,pa);    /* a*G =(pa,ya), ia is sign of ya */

    bigbits(mip,CURVE_BITS,b);  /* B's random number */
    ib=mul_brick(mip,&binst,b,pb,pb);    /* b*G =(pb,yb), ib is sign of yb */

/* online calculations */
    ecurve_init(mip,A,B,p,MR_PROJECTIVE);
    epoint_set(mip,pb,pb,ib,PB); /* decompress PB */
    ecurve_mult(mip,a,PB,PB);
    epoint_get(mip,PB,key,key);

/* since internal base is HEX, can use otnum instead of cotnum - avoiding a base change */
#ifndef MR_NO_STANDARD_IO
printf("Alice's Key= ");
otnum(mip,key,stdout);
#endif
    epoint_set(mip,pa,pa,ia,PB); /* decompress PA */
    ecurve_mult(mip,b,PB,PB);
    epoint_get(mip,PB,key,key);
#ifndef MR_NO_STANDARD_IO
printf("Bob's Key=   ");
otnum(mip,key,stdout);
#endif
/* clear the memory */

	memset(mem_big, 0, MR_BIG_RESERVE(10));
	memset(mem_ecp, 0, MR_ECP_RESERVE(2));

	return 0;
}
