/*
 *   Proposed Digital Signature Standard (DSS)
 *
 *   Elliptic Curve variation GF(2^m) - See Dr. Dobbs Journal, April 1997
 *
 *   This program asks for the name of a <file>, computes its message digest,
 *   signs it, and outputs the signature to a file <file>.ecs. It is assumed 
 *   that curve parameters are available from a file common2.ecs, as well as 
 *   the private key of the signer previously generated by the ecsgen2 program
 *
 *   The curve is y^2+xy = x^3+Ax^2+B over GF(2^m) using a trinomial or 
 *   pentanomial basis (t^m+t^a+1 or t^m+t^a+t^b+t^c+1), These parameters
 *   can be generated using the findbase.cpp example program, or taken from tables
 *   provided, for example in IEEE-P1363 Annex A 
 *
 *   The file common2.ecs is presumed to exist and contain 
 *   {m,A,B,q,x,y,a,b,c} where A and B are parameters of the equation 
 *   above, (x,y) is an initial point on the curve, {m,a,b,c} are the field 
 *   parameters, (b is zero for a trinomial) and q is the order of the 
 *   (x,y) point, itself a large prime. The number of points on the curve is 
 *   cf.q where cf is the "co-factor", normally 2 or 4.
 * 
 *   Copyright (c) 2000-2003 Shamus Software Ltd.
 */

#include <stdio.h>
#include "miracl.h"
#include <stdlib.h>
#include <string.h>

void strip(char *name)
{ /* strip off filename extension */
    int i;
    for (i=0;name[i]!='\0';i++)
    {
        if (name[i]!='.') continue;
        name[i]='\0';
        break;
    }
}

static void hashing(FILE *fp,big hash)
{ /* compute hash function */
    char h[20];
    int i,ch;
    sha sh;
    shs_init(&sh);
    while ((ch=fgetc(fp))!=EOF) shs_process(&sh,ch);
    shs_hash(&sh,h);
    bytes_to_big(20,h,hash);
}

int main()
{
    FILE *fp;
    int m,a,b,c;
    miracl *mip;
    char ifname[50],ofname[50];
    big a2,a6,q,x,y,d,r,s,k,hash;
    epoint *g;
    long seed;
/* get public data */
    fp=fopen("common2.ecs","rt");
    if (fp==NULL)
    {
        printf("file common2.ecs does not exist\n");
        return 0;
    }
    fscanf(fp,"%d\n",&m);

    mip=mirsys(MR_ROUNDUP(abs(m),4),16);  
    a2=mirvar(0);
    a6=mirvar(0);
    q=mirvar(0);
    x=mirvar(0);
    y=mirvar(0);
    d=mirvar(0);
    r=mirvar(0);
    s=mirvar(0);
    k=mirvar(0);
    hash=mirvar(0);

    innum(a2,fp);     /* curve parameters */
    innum(a6,fp);     /* curve parameters */

    innum(q,fp);     /* order of (x,y) */
    innum(x,fp);     /* (x,y) point on curve of order q */
    innum(y,fp);

    fscanf(fp,"%d\n",&a);
    fscanf(fp,"%d\n",&b);
    fscanf(fp,"%d\n",&c);
    fclose(fp);

/* randomise */
    printf("Enter 9 digit random number seed  = ");
    scanf("%ld",&seed);
    getchar();
    irand(seed);

    ecurve2_init(m,a,b,c,a2,a6,FALSE,MR_BEST);  /* initialise curve */

    g=epoint_init();
    epoint2_set(x,y,0,g); /* set point of order q */

/* calculate r - this can be done offline, 
   and hence amortized to almost nothing   */
    bigrand(q,k);
    ecurve2_mult(k,g,g);      /* see ebrick2.c for method to speed this up */
    epoint2_get(g,r,r);
    divide(r,q,q);

/* get private key of signer */
    fp=fopen("private.ecs","rt");
    if (fp==NULL)
    {
        printf("file private.ecs does not exist\n");
        return 0;
    }
    innum(d,fp);
    fclose(fp);

/* calculate message digest */
    printf("file to be signed = ");
    gets(ifname);
    strcpy(ofname,ifname);
    strip(ofname);
    strcat(ofname,".ecs");
    if ((fp=fopen(ifname,"rb"))==NULL)
    {
        printf("Unable to open file %s\n",ifname);
        return 0;
    }
    hashing(fp,hash);
    fclose(fp);

/* calculate s */
    xgcd(k,q,k,k,k);

    mad(d,r,hash,q,q,s);
    mad(s,k,k,q,q,s);
    fp=fopen(ofname,"wt");
    otnum(r,fp);
    otnum(s,fp);
    fclose(fp);
    return 0;
}

